//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.4

import wesual.data.Core        1.0
import wesual.data.FoodService 1.0

import wesual.Ui           1.0
import wesual.Ui.Private   1.0

UiPopupAnchor {
    id : flyout

    property Document currentProduct
    property FoodServiceAllergenFilter filter

    popup : Rectangle {
        id : wrapper

        property alias allergenListing : gridView.model
        property alias gridViewFocus : gridView.focus

        readonly property bool flipped : flyout.distanceBottom < preferredHeight
        readonly property bool shift   : flyout.distanceRight < wrapper.width
        readonly property int  preferredHeight : {
            if (gridView.model)
                return (gridView.model.length / 2) * gridView.cellHeight +
                        button.height + 30;
            return 0;
        }

        height : Math.min(
                     preferredHeight,
                     (flipped ?
                     flyout.distanceTop :
                     flyout.distanceBottom))

        anchors {
            left : parent.left
            leftMargin :
                shift ? Math.min(-(wrapper.width - flyout.distanceRight), 0) : 0
            top : flipped ? undefined : parent.bottom
            topMargin : flyout.parent.height
            bottom : flipped ? parent.top : undefined
        }

        opacity : 0

        states : [
            State {
                name : "visible"
                when : flyout.popupVisible

                PropertyChanges {
                    target : wrapper
                    opacity : 1
                }
            }
        ]

        transitions : Transition {
            NumberAnimation {
                property : "opacity"
                duration : 200
            }
        }

        width        : (2 * p_.maxCellWidth) + 8
        border.width : 1
        border.color : UiColors.getColor(UiColors.ControlGrey20)

        MouseArea {
            id : stealMouseHover

            anchors.fill : parent
            hoverEnabled : true
        }

        Item {
            width  : parent.width - 8
            height : parent.height - 28
            anchors {
                horizontalCenter : parent.horizontalCenter
                top : parent.top
                topMargin : 4
            }

            GridView {
                id : gridView

                anchors.fill : parent
                cellWidth : p_.maxCellWidth
                cellHeight : 42
                focus : true
                keyNavigationWraps : true
                interactive : false
                highlightFollowsCurrentItem : true

                onActiveFocusChanged : {
                    currentIndex = -1;
                    if (!activeFocus)
                        flyout.hide();
                }

                QtObject {
                    id : p_

                    property int maxCellWidth  : 0
                }

                delegate : Rectangle {
                    id : delegate

                    width  : p_.maxCellWidth
                    height : gridView.cellHeight
                    color  : UiColors.getColor(UiColors.White)

                    states : [
                        State {
                            name : "pressed"
                            when : ma.pressed || spacePressedTimer.running

                            PropertyChanges {
                                target : delegate
                                color  : UiColors.getColor(
                                             UiColors.SelectionGreen)
                            }
                            PropertyChanges {
                                target : allergenIcon
                                color  : UiColors.getColor(UiColors.White)
                            }
                            PropertyChanges {
                                target : allergenName
                                color  : UiColors.getColor(UiColors.White)
                            }
                        },
                        State {
                            name : "hovered"
                            when : index === gridView.currentIndex

                            PropertyChanges {
                                target : delegate
                                color  : UiColors.getColor(UiColors.HoverGreen)
                            }
                        }
                    ]

                    // key handling
                    Keys.onRightPressed  : {
                        if (gridView.currentIndex === -1)
                            gridView.currentIndex = 0;
                        else
                            gridView.moveCurrentIndexRight();
                    }
                    Keys.onLeftPressed   : {
                        if (gridView.currentIndex === -1)
                            gridView.currentIndex = 0;
                        else
                            gridView.moveCurrentIndexLeft();
                    }
                    Keys.onUpPressed     : {
                        if (gridView.currentIndex === -1)
                            gridView.currentIndex = 0;
                        else {
                            if (gridView.currentIndex === 1) {
                                gridView.moveCurrentIndexUp();
                                gridView.moveCurrentIndexLeft();
                            } else
                                gridView.moveCurrentIndexUp();
                        }
                    }
                    Keys.onDownPressed   : {
                        if (gridView.currentIndex === -1 ||
                            gridView.currentIndex === gridView.model.length - 1)
                            gridView.currentIndex = 0;
                        else {
                            gridView.moveCurrentIndexDown();

                            // wrap left
                            if (gridView.currentIndex === 1)
                                gridView.moveCurrentIndexLeft();
                            // wrap right
                            else if (gridView.currentIndex === 0)
                                gridView.moveCurrentIndexRight();
                        }
                    }

                    Keys.onSpacePressed  : {
                        spacePressedTimer.start();
                        selectAllergen();
                    }

                    Timer {
                        id : spacePressedTimer
                        interval : 100
                    }

                    Keys.onEnterPressed  : flyout.hide()
                    Keys.onReturnPressed : flyout.hide()

                    Row {
                        id : element
                        spacing : 5
                        height  : 30
                        anchors {
                            left : parent.left
                            verticalCenter: parent.verticalCenter
                            leftMargin : 5
                        }

                        onImplicitWidthChanged : {
                            p_.maxCellWidth =
                                 Math.max(p_.maxCellWidth, implicitWidth + 20);
                        }

                        UiCheckBox {
                            id : cb

                            anchors.verticalCenter : parent.verticalCenter
                            checked : {
                                if (!currentProduct && !filter)
                                    return false;

                                var tags;
                                if (currentProduct) {
                                    tags = currentProduct.tags;
                                } else if (filter) {
                                    tags = filter.tags;
                                }

                                for (var i = 0; i < tags.length; ++i) {
                                    if (modelData === tags[i])
                                        return true;
                                }

                                return false;
                            }
                        }

                        UiColorizedImage {
                            id : allergenIcon

                            width  : 34
                            height : 34
                            source : modelData.icon
                            color  : UiColors.getColor(UiColors.Black)

                            anchors.verticalCenter : parent.verticalCenter
                        }

                        Text {
                            id : allergenName

                            anchors.verticalCenter : parent.verticalCenter
                            text : modelData.name
                            font : UiFonts.getFont(UiFonts.Regular, 13)
                        }
                    }

                    MouseArea {
                        id : ma

                        anchors.fill      : parent

                        hoverEnabled      : true
                        onPositionChanged : gridView.currentIndex = index
                        onExited          : gridView.currentIndex = -1
                        onPressed         : selectAllergen()
                    }

                    function selectAllergen() {
                        if (!currentProduct && !filter)
                            return;

                        cb.checked = !cb.checked;
                        if (currentProduct) {
                            if (cb.checked)
                                currentProduct.addTag(modelData)
                            else
                                currentProduct.removeTag(modelData)
                        } else {
                            if (cb.checked)
                                filter.addTag(modelData)
                            else
                                filter.removeTag(modelData)
                        }
                    }
                }
            }
        }
        Rectangle {
            width  : parent.width
            height : 1
            color  : UiColors.getColor(UiColors.ControlGrey20)

            anchors.bottom       : button.top
            anchors.bottomMargin : 5
        }

        UiPushButton {
            id : button

            text : qsTrId("28c3843d05fb11e9")

            anchors {
                right        : parent.right
                rightMargin  : 12
                bottom       : parent.bottom
                bottomMargin : 6
            }

            onClicked : flyout.hide()
        }
    }
    onPopupVisibleChanged : {
        if (!popupVisible)
            addAllergen.forceActiveFocus();

        if (currentProduct) {
            if (popupVisible) {
                currentProduct.context.beginRecordChanges();
            } else {
                currentProduct.context.endRecordChanges();
            }
        }
    }

}
